(function ($) {
    'use strict';

    class SinCurve extends THREE.Curve {

        constructor(scale, t, deg) {
            super(scale, t, deg);
            this.scale = (scale === undefined) ? 1 : scale;
            this.t = (t === undefined) ? 10 : t;
            this.deg = (deg === undefined) ? 2 : deg;
        }

        getPoint(t, target_2) {

            var ty = t * this.t;
            var tx = Math.sin(this.deg * Math.PI * t);
            var tz = Math.cos(this.deg * Math.PI * t);

            var point = new THREE.Vector3(tx, ty, tz).multiplyScalar(this.scale);

            if (target_2) target_2.copy(point);

            return point;

        }

    }

    class DNA extends THREE.Group {

        constructor(curve, total, fresnelMat, fresnelMat2) {

            super();

            var cylLength = 1.65;
            var cylGeo = new THREE.CylinderBufferGeometry(.1, .1, cylLength / 2, 16, 1, true);
            var cylinder = new THREE.Mesh(cylGeo, fresnelMat);
            cylinder.position.y = cylLength / 4;

            var cylinder2 = new THREE.Mesh(cylGeo, fresnelMat2);
            cylinder2.position.y = -cylLength / 4;

            var sphereGeo = new THREE.SphereBufferGeometry(0.3, 32, 32);
            var sphere = new THREE.Mesh(sphereGeo, fresnelMat);
            sphere.position.y = cylLength / 2 + 0.25;

            var sphere2 = new THREE.Mesh(sphereGeo, fresnelMat2);
            sphere2.position.y = -cylLength / 2 - 0.25;

            var barGroup = new THREE.Group();
            barGroup.add(cylinder);
            barGroup.add(cylinder2);
            barGroup.add(sphere);
            barGroup.add(sphere2);

            total = total || 80;

            for (var i = 1; i <= total; i++) {

                var bGroup = new THREE.Group();

                var bar = barGroup.clone();
                bar.rotation.z = Math.PI * (i / 10);
                bar.userData.startZ = bar.rotation.z;
                bGroup.add(bar);

                curve.getPoint(i / total, bGroup.position);

                var nextPoint = curve.getPoint((i + 1) / total);
                bGroup.lookAt(nextPoint);

                this.add(bGroup);

            }
        }

        update(playhead) {
            this.children.forEach(obj => {
                if (obj.isGroup) {
                    var bar = obj.children[0];
                    bar.rotation.z = bar.userData.startZ - Math.PI * playhead;
                }
            });
        }
    }

    var VisualDNAAnimation = {
        initDNA: function () {
            elementorFrontend.hooks.addAction('frontend/element_ready/section', VisualDNAAnimation.initDNAWidget);
        },
        initDNAWidget: function ($scope) {
            var sectionId = $scope.data('id');
            var target = '.elementor-element-' + sectionId;
            var settings = {};
            if (window.isEditMode || window.elementorFrontend.isEditMode()) {
                var editorElements = null;
                var dnaAnimationArgs = {};

                if (!window.elementor.hasOwnProperty('elements')) {
                    return false;
                }

                editorElements = window.elementor.elements;

                if (!editorElements.models) {
                    return false;
                }

                $.each(editorElements.models, function (i, el) {
                    if (sectionId === el.id) {
                        dnaAnimationArgs = el.attributes.settings.attributes;
                    } else if (el.id === $scope.closest('.elementor-top-section').data('id')) {
                        $.each(el.attributes.elements.models, function (i, col) {
                            $.each(col.attributes.elements.models, function (i, subSec) {
                                dnaAnimationArgs = subSec.attributes.settings.attributes;
                            });
                        });
                    }
                    if (dnaAnimationArgs.marvy_enable_dna_animation) {
                        settings.switch = dnaAnimationArgs.marvy_enable_dna_animation;
                        settings.background_color = dnaAnimationArgs.marvy_dna_animation_background_color;
                        settings.speed = dnaAnimationArgs.marvy_dna_animation_speed;
                        settings.fill_color_1 = dnaAnimationArgs.marvy_dna_animation_fill_color_1;
                        settings.border_color_1 = dnaAnimationArgs.marvy_dna_animation_border_color_1;
                        settings.fill_color_2 = dnaAnimationArgs.marvy_dna_animation_fill_color_2;
                        settings.border_color_2 = dnaAnimationArgs.marvy_dna_animation_border_color_2;
                        settings.opacity = dnaAnimationArgs.marvy_dna_animation_opacity;
                        settings.brightness = dnaAnimationArgs.marvy_dna_animation_brightness;
                        settings.border_width = dnaAnimationArgs.marvy_dna_animation_border_width;
                        settings.is_transparent = dnaAnimationArgs.marvy_dna_animation_is_transparent;
                        settings.particles_counts = dnaAnimationArgs.marvy_dna_animation_particles_counts;
                        settings.particles_size = dnaAnimationArgs.marvy_dna_animation_particles_size;
                        settings.particles_color = dnaAnimationArgs.marvy_dna_animation_particles_color;
                        settings.dna_bond_count_1 = dnaAnimationArgs.marvy_dna_animation_dna_bond_count_1;
                        settings.dna_bond_count_2 = dnaAnimationArgs.marvy_dna_animation_dna_bond_count_2;
                        settings.dna_bond_count_3 = dnaAnimationArgs.marvy_dna_animation_dna_bond_count_3;
                    }
                });

            } else {
                settings.switch = $scope.data("marvy_enable_dna_animation");
                settings.background_color = $scope.data("marvy_dna_animation_background_color");
                settings.speed = $scope.data("marvy_dna_animation_speed");
                settings.fill_color_1 = $scope.data("marvy_dna_animation_fill_color_1");
                settings.border_color_1 = $scope.data("marvy_dna_animation_border_color_1");
                settings.fill_color_2 = $scope.data("marvy_dna_animation_fill_color_2");
                settings.border_color_2 = $scope.data("marvy_dna_animation_border_color_2");
                settings.opacity = $scope.data("marvy_dna_animation_opacity");
                settings.brightness = $scope.data("marvy_dna_animation_brightness");
                settings.border_width = $scope.data("marvy_dna_animation_border_width");
                settings.is_transparent = $scope.data("marvy_dna_animation_is_transparent");
                settings.particles_counts = $scope.data("marvy_dna_animation_particles_counts");
                settings.particles_size = $scope.data("marvy_dna_animation_particles_size");
                settings.particles_color = $scope.data("marvy_dna_animation_particles_color");
                settings.dna_bond_count_1 = $scope.data("marvy_dna_animation_dna_bond_count_1");
                settings.dna_bond_count_2 = $scope.data("marvy_dna_animation_dna_bond_count_2");
                settings.dna_bond_count_3 = $scope.data("marvy_dna_animation_dna_bond_count_3");
            }
            if (settings.switch) {
                dnaAnimation(target, settings, sectionId);
            }
        }
    };

    function getStyle(el) {
        return (typeof getComputedStyle !== 'undefined' ?
                getComputedStyle(el, null) :
                el.currentStyle
        );
    }

    function dnaAnimation(target, settings, sectionId) {
        var checkElement = document.getElementsByClassName("marvy-dna-section-" + sectionId);
        if (checkElement.length >= 0 && settings) {
            document.querySelector(target).style.overflow = "hidden !important";
            document.querySelector(target).classList.add("marvy-custom-dna-animation-section-" + sectionId);

            var speed = 1000 * settings.speed;
            var is_transparent = settings.is_transparent === 'yes' ? true : false;

            var dna_div = document.createElement('div');
            dna_div.setAttribute('id', 'marvy-dna-container-' + sectionId);
            dna_div.classList.add("marvy-dna-section-" + sectionId);
            document.querySelector(target).appendChild(dna_div);

            // Set Z-index for section container
            var dnaZindex = document.querySelector('.marvy-custom-dna-animation-section-' + sectionId + ' .elementor-container');
            dnaZindex.style.zIndex = '99';
            dnaZindex.style.width = '100%';

            var target_style = getStyle(document.querySelector(target));

            var camera = new THREE.PerspectiveCamera(60, parseInt(target_style['width']) / (parseInt(target_style['height']) > 0 ? parseInt(target_style['height']) : 1), 0.01, 1000);
            var renderer = new THREE.WebGLRenderer({antialias: true});
            var scene = new THREE.Scene();
            var target_1 = new THREE.Vector3();

            var curve1 = new SinCurve(4.5, 10, 2.5);
            var curve2 = new SinCurve(6, 10, 2);
            var curve3 = new SinCurve(4, 15, 2.8);

            renderer.setPixelRatio(window.devicePixelRatio);
            renderer.setSize(parseInt(target_style['width']), parseInt(target_style['height']));
            dna_div.appendChild(renderer.domElement);

            renderer.setClearColor('#000000', 1);

            scene.background = new THREE.Color(settings.background_color);

            camera.position.set(0, 0, 25);


            var fresnelMat = new THREE.ShaderMaterial({

                uniforms: {
                    color1: {type: "c", value: new THREE.Color(settings.border_color_1)}, // edge, light blue
                    color2: {type: "c", value: new THREE.Color(settings.fill_color_1)}, // base, dark blue
                    alpha: {type: "f", value: settings.opacity},
                    fresnelBias: {type: "f", value: settings.brightness},
                    fresnelScale: {type: "f", value: settings.border_width},
                    fresnelPower: {type: 'f', value: 1.5}
                },

                vertexShader: /* glsl */ `
    uniform float fresnelBias;
    uniform float fresnelScale;
    uniform float fresnelPower;

    varying float vReflectionFactor;

    void main() {
      vec4 mvPosition = modelViewMatrix * vec4( position, 1.0 );
      vec4 worldPosition = modelMatrix * vec4( position, 1.0 );

      vec3 worldNormal = normalize( mat3( modelMatrix[0].xyz, modelMatrix[1].xyz, modelMatrix[2].xyz ) * normal );

      vec3 I = worldPosition.xyz - cameraPosition;

      vReflectionFactor = fresnelBias + fresnelScale * pow( 1.0 + dot( normalize( I ), worldNormal ), fresnelPower );

      gl_Position = projectionMatrix * mvPosition;
    }
  `,

                fragmentShader: /* glsl */ `
    uniform vec3 color1;
    uniform vec3 color2;
    uniform float alpha;

    varying float vReflectionFactor;

    void main() {
      gl_FragColor = vec4(mix(color2, color1, vec3(clamp( vReflectionFactor, 0.0, 1.0 ))), alpha);
    }
  `,
                transparent: is_transparent,
            });

            var fresnelMat2 = fresnelMat.clone();
            fresnelMat2.uniforms.color1.value.set(settings.border_color_2);
            fresnelMat2.uniforms.color2.value.set(settings.fill_color_2);

            var dna1 = window.dna1 = new DNA(curve1, settings.dna_bond_count_1, fresnelMat, fresnelMat2);
            var dna2 = new DNA(curve2, settings.dna_bond_count_2, fresnelMat, fresnelMat2);
            var dna3 = new DNA(curve3, settings.dna_bond_count_3, fresnelMat, fresnelMat2);

            scene.add(dna1);
            dna1.position.set(1, -21, 13);


            scene.add(dna2);
            dna2.position.set(10, -30, -4);

            scene.add(dna3);
            dna3.position.set(-10, -28, -4);

            var ParticleShader = {

                uniforms: {
                    color: {type: 'v3', value: new THREE.Color(settings.particles_color)},
                    texture: {type: 't', value: null},
                    time: {type: 'f', value: 0},
                    size: {type: 'f', value: 50.0}
                },

                vertexShader: /* glsl */ `
    uniform float time;
    uniform float size;
    attribute float alphaOffset;
    varying float vAlpha;
    uniform vec4 origin;

    void main() {

      vAlpha = 0.5 * ( 1.0 + sin( alphaOffset + time ) );
      // vAlpha = 1.0;

      vec4 mvPosition = modelViewMatrix * vec4( position, 1.0 );
      float cameraDist = distance( mvPosition, origin );
      gl_PointSize = size / cameraDist;
      gl_Position = projectionMatrix * mvPosition;

    }
  `,

                fragmentShader: /* glsl */ `
    uniform float time;
    uniform vec3 color;

    varying float vAlpha;

    void main() {
      vec2 center = gl_PointCoord - 0.5;
      float dist = length(center);
      float alpha = smoothstep(0.5, 0.1, dist) * vAlpha;
      gl_FragColor = vec4( color, alpha );
    }
  `
            };

            var particles = new Particles({
                color: 0xffffff,
                range: new THREE.Vector3(50, 50, 50),
                pointCount: settings.particles_counts,
                size: settings.particles_size,
                speed: 0.1
            }, ParticleShader);

            scene.add(particles);

            var timeline = {
                playhead: 0
            }

            var tween1 = new TWEEN.Tween(timeline)
                .to({playhead: 1}, speed)
                .easing(TWEEN.Easing.Linear.None)
                .repeat(Infinity)
                .start();

            let resizeObserver = new ResizeObserver((entries) => {
                if (renderer !== undefined) {
                    resize(renderer, target, camera);
                }
            });

            resizeObserver.observe(document.querySelector(target));

            renderer.setAnimationLoop(() => {
                loop(renderer, timeline, dna1, dna2, dna3, camera, target_1, scene)
            });

        }
        return true;
    }

    function resize(renderer, target, camera) {
        if(document.querySelector(target)) {
            var target_style = getStyle(document.querySelector(target));
            camera.aspect = parseInt(target_style['width']) / (parseInt(target_style['height']) > 0 ? parseInt(target_style['height']) : 1);
            camera.updateProjectionMatrix();
            renderer.setSize(parseInt(target_style['width']), parseInt(target_style['height']));
        }
    }

    function loop(renderer, timeline, dna1, dna2, dna3, camera, target_1, scene) {

        TWEEN.update();

        var playhead = timeline.playhead;

        dna1.update(playhead * 8);
        dna2.update(playhead * 8);
        dna3.update(playhead * 8);

        camera.position.x = -Math.sin(2 * Math.PI * playhead) * 25;
        camera.position.z = Math.cos(2 * Math.PI * playhead) * 25;
        camera.position.y = Math.sin(2 * 2 * Math.PI * playhead) * 5;

        target_1.x = -Math.sin(2 * Math.PI * playhead) * 10;
        target_1.z = Math.cos(2 * Math.PI * playhead) * 10;

        camera.lookAt(target_1);

        renderer.render(scene, camera);

    }

    class Particles extends THREE.Group {

        constructor(options, ParticleShader) {

            super();

            var color = this.color = options.color || 0x333333;
            var size = this.size = options.size || 0.4;

            var pointCount = this.pointCount = options.pointCount || 40;
            var range = this.range = options.range || new THREE.Vector3(2, 2, 2);

            ParticleShader.uniforms.size.value = size;

            var pointsMat = new THREE.ShaderMaterial({
                uniforms: ParticleShader.uniforms,
                vertexShader: ParticleShader.vertexShader,
                fragmentShader: ParticleShader.fragmentShader,
                blending: THREE.AdditiveBlending,
                depthWrite: false,
                transparent: true
            });

            var pointsGeo = new THREE.BufferGeometry();

            var positions = new Float32Array(pointCount * 3);
            var alphas = new Float32Array(pointCount);

            for (var i = 0; i < pointCount; i++) {

                positions[i * 3 + 0] = THREE.Math.randFloatSpread(range.x);
                positions[i * 3 + 1] = THREE.Math.randFloatSpread(range.y);
                positions[i * 3 + 2] = THREE.Math.randFloatSpread(range.z);

                alphas[i] = i;

            }

            pointsGeo.addAttribute('position', new THREE.BufferAttribute(positions, 3));
            pointsGeo.addAttribute('alphaOffset', new THREE.BufferAttribute(alphas, 1));

            var points = this.points = new THREE.Points(pointsGeo, pointsMat);
            points.sortParticles = true;
            points.renderOrder = 1;

            this.add(points);

            var box = new THREE.Box3();
            box.setFromCenterAndSize(points.position, range);

        }

    }

    $(window).on('elementor/frontend/init', VisualDNAAnimation.initDNA);
})(jQuery);
